########
# Author: Matthew E Craig
# This script makes and saves Figures 2 and S1 from Craig et al. 2021 (Global Change Biology)
########

#housekeeping
rm(list=ls())

#load packages
packload <- c('tidyverse',    #dplyr and ggplot2 packages needed
              'standardize',  #functions for calculating z-scores to standardize MBC data
              'cowplot'       #used for arranging ggplot panels
              )
lapply(packload, library, character.only=TRUE)

#read in data
mbdat <- read_csv('MBC_data_compilation_20210309.csv') %>% 
  mutate_at(vars(Study,ID), list(factor)) %>% #specify column data types that were incorrectly 'guessed'
  filter(Study != 10) %>% #removing Brown study which is confounded with N additions
  mutate(c_lev_z = scale_by(c_lev~ID), #creating z-transformed carbon input and microbial biomass variables
         mbc_z = scale_by(mbc~ID))

#plotting function
make_MBC_plot <- function(df){
  ggplot(df, aes(x=c_lev_z, y=mbc_z))+
    geom_point(alpha=.5, size = 2.5, color = 'grey50')+
    #extrapolate linear relationship from bottom half of the input rate data
    stat_smooth(data = filter(df, c_lev_z <= 0 ), method = "lm", se=FALSE, inherit.aes=FALSE, 
                aes(x=c_lev_z, y=mbc_z), fullrange = TRUE, color='black', size = 1, linetype = 'dashed')+
    #fit loess smoothing function to data
    stat_smooth(method = 'loess', formula = y ~x, span = 1, se=TRUE, inherit.aes = FALSE, 
                aes(x=c_lev_z,y=mbc_z), color='black', size=2)+
    labs(x='Organic input rate (standardized)', y='Microbial biomass-C (standardized)')+
    theme_bw()+
    theme(legend.position='none',
          panel.grid = element_blank()) 
}

#Fig. 2: complete dataset
data_synth_fig <- make_MBC_plot(mbdat)

#figS1A: only studies that were run for more than 1 year
data_synth_fig_old <- make_MBC_plot(mbdat %>% filter(median_dur >= 1))

#figS1B: only field studies
data_synth_fig_natural <- make_MBC_plot(mbdat %>% filter(System != 'lab'))

#FigS1C: only studies where microbial respiration was also measured
#Study IDs for such papers based on Table S1
CO2_papers <- c('12','21','19','17','35','29','15','41','7','13','16','11','54','38','30') 
data_synth_fig_CO2 <- make_MBC_plot(filter(mbdat, Study %in% CO2_papers))

#construct supplementary figure
supp_data_synth_fig <- plot_grid(data_synth_fig_old + xlab(""), data_synth_fig_natural + xlab(""), 
                                 data_synth_fig_CO2, labels = c('(a)','(b)','(c)'), hjust = -.1, ncol = 1)

#output Fig. 2
tiff('Fig_2.tiff', res = 300, width = 10, height = 10, units = 'cm')
data_synth_fig
dev.off()

#output Fig S1
tiff('supp_data_synth_fig.tiff', res = 100, width = 10, height = 28, units = 'cm')
supp_data_synth_fig
dev.off()
